#!/usr/bin/env python3
#-----------------------------------------------------------------------------
# Copyright (C) 2017 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
# 
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
# 
#
# This program computes the volume integrals (reference and deformed) for the
# top and bottom side given two masks generated with the mask_dt program.

from VicPy import *

import numpy as np
import sys
from os import _exit as exit
import os
import glob

def masked_integral(aoi, mask):
    nr = aoi.numRows()
    nc = aoi.numColumns()
    if mask.shape != (nr, nc):
        raise ValueError('Mask size differs from AOI size.')
    val = aoi.asArray(['sigma', 'V0', 'V1'])
    s = np.reshape(val['sigma'], (nr, nc))
    mul = np.logical_and(mask > 0, s >= 0.0) * np.float(1.0)
    v0 = np.reshape(val['V0'], (nr, nc)) * mul
    v1 = np.reshape(val['V1'], (nr, nc)) * mul

    return np.sum(v0), np.sum(v1)

if __name__ == '__main__':
    if len(sys.argv) < 3:
        print('Usage: {0} mask-prefix input_0.out [input_1.out ...]'.format(sys.argv[0]), file=sys.stderr)
        exit(-1)
    
    prefix = sys.argv[1]
    try:
        mt = np.load('{0}_top.npy'.format(prefix))
        mb = np.load('{0}_bot.npy'.format(prefix))
    except Exception as e:
        print("Could not load masks: " + str(e), file=sys.stderr)
        exit(-1)
    log = open('volumina.log', 'w')
    msg = '"V0 top" "V1 top" "V0 bot" "V1 bot" "dV/V0 top" "dV/V0 bot"'
    print(msg)
    print(msg, file=log)

    input_files = sys.argv[2:]
    if len(input_files) == 1:
        if not os.path.isfile(input_files[0]):
            input_files = glob.glob(input_files[0])
    for filename in input_files:
        # Create data set and load the data
        data = VicDataSet()
        if data.load(filename) == False:
            print("Could not load dataset from file {0}".format(filename))
            continue
        # Check that two AOIs are present
        if data.numData() != 2:
            print("Data must have exactly two aois.", file=sys.stderr)
            continue
        try:
            v0_t, v1_t = masked_integral(data.data(0), mt)
            v0_b, v1_b = masked_integral(data.data(1), mb)
        except Exception as e:
            print("Integral failed: " + str(e), file=sys.stderr)
            continue
        msg = '{0} {1} {2} {3} {4} {5}'.format(v0_t, v1_t, v0_b, v1_b, v1_t / v0_t - 1, v1_b / v0_b - 1)
        print(msg)
        print(msg, file=log)
    log.close()
