#!/usr/bin/env python3
#-----------------------------------------------------------------------------
# Copyright (C) 2017 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
# 
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
# 
#
# hexahedron volume computation
import os
import sys
import numpy as np
import math

def hexahedron_volume(points):
    # This routine computes the (unsigned) volume of
    # a hexahedron given by two quadrilaterals.
    #
    # The first four vertices correspond to the bottom
    # surface, the second four to the top surface.
    # Each surface has to be specified in the same
    # order (cw or ccw) to obtain the correct volume.
    # For a cube of size 2 * (width * height * depth)
    # centered around (0, 0, 0), the vertices would be:
    #      -width, -height, -depth, // 0
    #       width, -height, -depth, // 1
    #       width,  height, -depth, // 2
    #      -width,  height, -depth, // 3
    #      -width, -height,  depth, // 4
    #       width, -height,  depth, // 5
    #       width,  height,  depth, // 6
    #      -width,  height,  depth  // 7
    #
    # The top/bottom surface can be swapped, as long
    # as they are both specified in the same order
    # (cw or ccw).
    # 

    faces = [[0, 2, 1],
             [0, 3, 2],

             [1, 2, 6],
             [6, 5, 1],

             [4, 5, 6],
             [6, 7, 4],

             [2, 3, 6],
             [6, 3, 7],

             [0, 7, 3],
             [0, 4, 7],

             [0, 1, 5],
             [0, 5, 4]]

    v = 0
    for face in faces:
        v += np.dot(np.cross(points[face[0]], points[face[1]]), points[face[2]])
    return math.fabs(v / 6.0)

if __name__ == '__main__':
    if len(sys.argv) != 2:
        print("Usage: " + sys.argv[0] + " filename")
        os._exit(-1)

    points = np.loadtxt(sys.argv[1])
    if len(points) != 8:
        print("Wrong number of points in the file. Must be 8.")
        os.exit(-1)

    v = hexahedron_volume(points)
    print("Hexahedron volume is %f" % v)

