#!/usr/bin/env python3
#-----------------------------------------------------------------------------
# Copyright (C) 2017 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
# 
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
# 
#
# This program computes a mask for the top and bottom side of a surface based
# on a specified threshold for the variable dt.
# The masks are saved as two numpy arrays.

from VicPy import *

import numpy as np
import sys
from os import _exit as exit

def thresh_dt(aoi, thresh):
    nr = aoi.numRows()
    nc = aoi.numColumns()
    val = aoi.asArray(['sigma', 'dt'])
    s = np.reshape(val['sigma'], (nr, nc))
    t = np.reshape(val['dt'], (nr, nc))
    mask = np.logical_and(s >= 0.0, t <= thresh) * np.uint8(255)
    return mask

if __name__ == '__main__':
    if len(sys.argv) != 4:
        print("Usage: %s input.out threshold output_prefix\n" % sys.argv[0])
        exit(-1)

    thresh = float(sys.argv[2])
    data = VicDataSet()
    if not data.load(sys.argv[1]):
        print('Could not load data from file {0}'.format(sys.argv[1]), file=sys.stderr)
        exit(-1)
    if data.numData() != 2:
        print("Data must have exactly two aois.", file=sys.stderr)
        exit(-1)
    try:
        mt = thresh_dt(data.data(0), thresh)
        mb = thresh_dt(data.data(1), thresh)
    except ValueError as e:
        print("Masking failed: " + str(e), file=sys.stderr)
        exit(-1)
    prefix = sys.argv[3]
    np.save('{0}_top.npy'.format(prefix), mt)
    np.save('{0}_bot.npy'.format(prefix), mb)

    try:
        from PIL import Image
    except:
        print('The PIL library is not installed, skipping image generation.', file=sys.stderr)
        exit(0)

    it = Image.fromarray(mt, 'L')
    ib = Image.fromarray(mb, 'L')
    it.save('{0}_top.png'.format(prefix))
    ib.save('{0}_bot.png'.format(prefix))
