#!/usr/bin/env python3
#-----------------------------------------------------------------------------
# Copyright (C) 2020 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
# 
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
#-----------------------------------------------------------------------------
# This program illustrates how to read an stl file and convert
# it to a vtp file.
#-----------------------------------------------------------------------------

import time
from io import open
from os import _exit as exit
import sys
import vtk
import numpy as np
from stl import mesh

def stl2vtk(data):
    surface = vtk.vtkPolyData()
    points = vtk.vtkPoints()
    polys = vtk.vtkCellArray()

    s = data.vectors.shape
    n = s[0]
    X = data.vectors.reshape([s[0] * s[1], s[2]])
    for i, p in enumerate(X):
        points.InsertPoint(i, p)

    for i in range(n):
        polys.InsertNextCell(3, [3*i, 3*i+1, 3*i+2])
    
    surface.SetPoints(points)
    surface.SetPolys(polys)
    
    return surface

def save_vtk(object, filename):
    writer = vtk.vtkXMLPolyDataWriter()
    writer.SetFileName(filename)
    writer.SetInputData(object)
    writer.Write()


if __name__ == '__main__':
    
    t0 = time.time()
    
    if(len(sys.argv) != 3):
        print("Usage: %s infile.stl outfile.vtp\n" % sys.argv[0])
        exit(-1)
        
    data = mesh.Mesh.from_file(sys.argv[1])
    surface = stl2vtk(data)
    save_vtk(surface, sys.argv[2])
    
    t1 = time.time()
    print("Time for data conversion: {0:.3f} seconds.\n".format(t1-t0))
