#!/usr/bin/env python3
# -----------------------------------------------------------------------------
# Copyright (C) 2017 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
# 
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
# 
#
# This program displays the distribution of values for a user selectable
# variable in a Vic data file as a histogram.

from __future__ import unicode_literals

import pathlib
import sys
import os
from zipfile import PyZipFile

import matplotlib

# Make sure that we are using QT5
from PyQt5.uic import loadUi
from matplotlib.ticker import FuncFormatter

matplotlib.use('Qt5Agg')
from PyQt5 import QtCore, QtWidgets, QtGui

from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
from matplotlib.figure import Figure
from VicPy import *
import numpy as np
import lxml.etree as etree

class HistogramWidget(FigureCanvas):
    def __init__(self, parent=None, width=5, height=4, dpi=100):
        self.fig = Figure(figsize=(width, height), dpi=dpi)
        FigureCanvas.__init__(self, self.fig)
        self.axes = self.fig.add_subplot(111)
        self.setParent(parent)
        FigureCanvas.setSizePolicy(self,
                                   QtWidgets.QSizePolicy.Expanding,
                                   QtWidgets.QSizePolicy.Expanding)
        FigureCanvas.updateGeometry(self)

    def build_histogram(self, values, bins, min_v, max_v, description):
        self.axes.cla()
        if len(values) == 0:
            self.axes.plot([])
        else:
            # self.axes.hist(values, bins=bins)
            ys, xs = np.histogram(values, bins, range=(min_v, max_v))
            ys = ys / np.sum(ys)
            self.axes.bar(xs[0:len(ys)], ys, (max_v - min_v) / bins)
            self.axes.set_xlim((min_v, max_v))
            self.axes.set_xlabel(description)

        self.draw()


class ApplicationWindow(QtWidgets.QMainWindow):
    def __init__(self):
        QtWidgets.QMainWindow.__init__(self)
        # load widget from ui
        dir = os.path.dirname(os.path.abspath(__file__))
        
        loadUi(os.path.join(dir, 'mainwindow.ui'), self)

        self._settings = QtCore.QSettings("Correlated Solutions, Inc.", "Histogram")
        self._files = set()
        self._sorted_full_filenames = []
        self._vars = {}
        self._var_description = ""
        self._data = None
        self._values = None

        self.setAttribute(QtCore.Qt.WA_DeleteOnClose)

        # connect menu items
        self.actionQuit.triggered.connect(self.file_quit)
        self.actionAdd_Files.triggered.connect(self.open_files)
        self.actionAdd_Folder.triggered.connect(self.open_folder)
        self.actionOpen_Project_File.triggered.connect(self.open_project_file)
        self.actionClear_File_List.triggered.connect(self.clear_file_list)

        # connect model to filelist
        self._filelist_model = QtCore.QStringListModel(self.filelist)
        self.filelist.setModel(self._filelist_model)
        self.filelist.activated.connect(self.on_file_selected)

        # set up variables combobox
        self.var.activated.connect(self.on_variable_selected)

        # set limitations on input data in edits (positive integers)
        # self.bins.setValidator(QtGui.QIntValidator(10, 100, self))
        self.min.setValidator(QtGui.QDoubleValidator(self))
        self.max.setValidator(QtGui.QDoubleValidator(self))

        self.bins.valueChanged.connect(self.show_histogram)
        self.min.editingFinished.connect(self.show_histogram)
        self.max.editingFinished.connect(self.show_histogram)

        # add matplotlib widget
        self._hw = HistogramWidget(self.centralWidget, width=5, height=4, dpi=100)
        self.vl.addWidget(self._hw)

        # self.update_filelist()

    def _set_folder(self, folder):
        self._settings.setValue("path", folder)

    def _get_folder(self):
        return self._settings.value("path")

    def on_file_selected(self, i):
        filename = self._sorted_full_filenames[i.row()]
        self._data = VicDataSet()
        self._data.load(filename)

        self._data.computeStrain(tensorType="hencky", windowSize=7, outputGradient=False,
                              computePrincipalStrain=True, reuseVars=True,
                              weightedFilter=True, computeTresca=False, computeVonMises=False)

        # get all variable names
        self._vars = {}
        for i in range(self._data.numData()):
            d = self._data.data(i)
            for j in range(d.numVariables()):
                self._vars[d.varName(j)] = d.varDescription(j)

        var_names = list(self._vars)
        var_names.sort()
        selected_text = self.var.currentText()
        # set variable names into combobox
        self.var.clear()
        self.var.insertItems(0, var_names)
        self.var.setCurrentText(selected_text)
        self.on_variable_selected(0)

    def on_variable_selected(self, i):
        var_name = self.var.currentText()
        self._var_description = self._vars[var_name]
        values = []
        for i in range(self._data.numData()):
            d = self._data.data(i)
            idx = d.varIndex(var_name)
            s_idx = d.varIndex("sigma")
            for j in range(d.matrixSize()):
                if d.value(j, s_idx) >= 0:
                    values.append(d.value(j, idx))
        values = np.asarray(values)
        self._values = values[np.isfinite(values)]
        self._values.sort()
        maximum, minimum = 0., 0.
        if len(self._values) > 0:
            maximum = np.max(self._values)
            minimum = np.min(self._values)

        self.min.setText(str(minimum))
        self.max.setText(str(maximum))
        self.show_histogram()

    def show_histogram(self):
        if not self.min.text() or not self.max.text():
            return
        min_v = float(self.min.text())
        max_v = float(self.max.text())
        values = self._values[(min_v <= self._values) & (self._values <= max_v)]
        self._hw.build_histogram(values, int(self.bins.value()), min_v, max_v, self._var_description)

    def file_quit(self):
        self.close()

    def update_file_list(self):
        self._sorted_full_filenames = []
        for f in self._files:
            self._sorted_full_filenames.append(f)
        self._sorted_full_filenames.sort()
        sorted_short_filenames = []
        for f in self._sorted_full_filenames:
            sorted_short_filenames.append(os.path.basename(f))
        self._filelist_model.setStringList(sorted_short_filenames)

    def add_files(self, files):
        for f in files:
            if os.path.exists(str(f)):
                self._files.add(str(f))

        self.update_file_list()
        # update file list function

    def open_files(self):
        files = QtWidgets.QFileDialog.getOpenFileNames(self.centralWidget,
                                                       "Add Vic-3D binary files",
                                                       self._get_folder(),
                                                       "Binaries (*.out)")[0]
        if len(files) > 0:
            self._set_folder(str(os.path.dirname(files[0])))
            self.add_files(files)

    def open_folder(self):
        fd = QtWidgets.QFileDialog(self.centralWidget)
        fd.setFileMode(QtWidgets.QFileDialog.Directory)
        fd.setDirectory(self._get_folder())
        folder_name = ''
        if fd.exec():
            folder_name = fd.selectedFiles()[0]
        if folder_name != '':
            self._set_folder(str(os.path.dirname(folder_name)))
            files = pathlib.Path(folder_name).glob('*.out')
            self.add_files(files)

    def open_project_file(self):
        project_file = QtWidgets.QFileDialog.getOpenFileName(self.centralWidget,
                                                             "Open Vic-3D project file",
                                                             self._get_folder(),
                                                             "Vic-3D project file (*.z3d)")[0]

        if project_file == '':
            return

        self.clear_file_list()
        self._set_folder(str(os.path.dirname(project_file)))

        p = PyZipFile(project_file, 'r')
        parser = etree.XMLParser(remove_blank_text=True)
        root = etree.fromstring(p.read("project.xml"), parser)
        path = root.get('dir')
        data_file_path = '//files/datafile'
        # collect all data files
        files = []
        for f in root.xpath(data_file_path):
            files.append(os.path.join(path, f.text))

        if len(files) > 0:
            self.add_files(files)

    def clear_file_list(self):
        self._files.clear()
        self.update_file_list()

    def closeEvent(self, ce):
        self.file_quit()


def main():
    q_app = QtWidgets.QApplication(sys.argv)
    aw = ApplicationWindow()
    aw.show()
    sys.exit(q_app.exec_())


if __name__ == '__main__':
    main()
