#!/usr/bin/env python3
#-----------------------------------------------------------------------------
# Copyright (C) 2017 Correlated Solutions, Inc.
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
# LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
# OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
#-----------------------------------------------------------------------------
# This program computes the acceleration for a series of data files.
#-----------------------------------------------------------------------------

from VicPy import *
from PIL import Image
import sys
import glob
import time
from os import _exit as exit

t0 = time.time()
data = VicDataSet()

if len(sys.argv) != 3:
    print("Usage: %s prefix time_constant\n" % sys.argv[0])
    exit(-1)

input_files = glob.glob(sys.argv[1] + "*_0.out")
input_files.sort()

dataL = VicDataSet()
dataR = VicDataSet()
dataOut = VicDataSet()

#check if 3D
dataIs3D = True
tempDataSet = VicDataSet()
tempDataSet.load(input_files[0])
tempData = tempDataSet.data(0)
if(tempData.varIndex("dW_dt")<0):
    dataIs3D = False
if(tempData.varIndex("dU_dt")<0):
    print("Velocity data not found. Please calculate the velocity first.")
    exit(-1)

if(dataIs3D):
    in_names = ["dU_dt", "dV_dt", "dW_dt"]
    out_names = ["Upp", "Vpp", "Wpp"]
    out_desc = ["d2U_dt2", "d2V_dt2", "d2W_dt2"]
else:
    in_names = ["dU_dt", "dV_dt"]
    out_desc = ["d2U_dt2", "d2V_dt2"]
    out_names = ["Upp", "Vpp"]

aoi = 0
time_constant = float(sys.argv[2])

print("Processing with time delta of %f." % time_constant)
n_files = len(input_files)

for i in range(n_files-1):
    div = 1
    if i==0:
        leftFile = input_files[0]
        rightFile = input_files[1]
    elif i==n_files-1:
        leftFile = input_files[n_files-2]
        rightFile = input_files[n_files-1]
    else:
        leftFile = input_files[i-1]
        rightFile = input_files[i+1]
        div = 2
    dataL.load(leftFile)
    dataR.load(rightFile)
    dataOut.load(input_files[i])
    print("Processing %s." % input_files[i])
    if dataL.numData()!=dataR.numData():
        print("  AOI count does not match between %s and %s." % (leftFile, rightFile))
        continue
    if dataL.numData()!=dataOut.numData():
        print("  AOI count does not match between %s and %s." % (leftFile, input_files[i]))
        continue
    for varct in range(len(out_names)):
        dataOut.addVariable(out_names[varct], out_desc[varct])
    for aoi in range(dataL.numData()):
        dL = dataL.data(aoi)
        dR = dataR.data(aoi)
        dO = dataOut.data(aoi)
        if(dL.matrixSize()!=dR.matrixSize()):
            print("  Matrix size mismatch - AOI sizes are not identical.")
            continue
        if(dL.matrixSize()!=dO.matrixSize()):
            print("  Matrix size mismatch - AOI sizes are not identical.")
            continue
        in_ids = []
        out_ids = []

        for var in in_names:
            idx = dL.varIndex(var)
            in_ids.append(idx)

        for var in out_names:
            idx = dO.varIndex(var)
            out_ids.append(idx)

        for pt_idx in range(dL.matrixSize()):
            valuesL = dL.values(pt_idx, in_ids)
            valuesR = dR.values(pt_idx, in_ids)
            for j in range(len(in_ids)):
                accel = valuesR[j]-valuesL[j]
                accel /= div
                accel /= time_constant;
                dO.setValue(pt_idx, out_ids[j], accel)

    if dataOut.save(input_files[i]) == False:
        print("Could not save the dataset.")
        exit(-1)

t1 = time.time()
print("Time for computing acceleration: {0:.3f} seconds.\n".format(t1-t0))
